﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.Security.Principal;
using System.ServiceModel;
using System.ServiceModel.Security;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Web;
using System.Web.Helpers;
using System.Web.Mvc;
using System.Web.Script.Serialization;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using BMS.FaultContracts;
using BMS.Utils;
using BMS.Utils.Properties;
using BMS.VistaIntegration.FacadeContracts.Admin;
using BMS.Web.App_GlobalResource;
using BMS.Web.Controllers.Shared;
using BMS.Web.Models;
using BMS.Web.Models.Shared;
using InfoWorld.HL7.ITS;
using VI = BMS.VistaIntegration.FacadeContracts;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Base Controller class.
    /// </summary>
    [LoggedUserFilter]
    [BMS.Web.Controllers.BaseController.ValidateAntiForgeryTokenWrapper(HttpVerbs.Post)]
    public class BaseController : Controller
    {
        protected User loggedUser;

        protected Dictionary<string, string> QueryStrings;

        public string FullUserName { get { return GetFullUserName(); } }

#if !DEBUG
        /// <summary>
        /// Called when an unhandled exception occurs in the action. 
        /// On SecurityNegotiationException force a re-logon.
        /// </summary>
        /// <param name="filterContext">Information about the current request and action.</param>
        protected sealed override void OnException(ExceptionContext filterContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                base.OnException(filterContext);
                if (IsSessionCriticalException(filterContext.Exception))
                {
                    RedirectLogin(filterContext.HttpContext);
                    return;
                }

                ActionResult ar = OnException(filterContext.Exception);
                if (ar != null)
                {
                    filterContext.Result = ar;
                    filterContext.ExceptionHandled = true;
                    return;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
#endif

        internal ActionResult UnauthorizedUrlRequestResult()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return UnauthorizedUrlRequestResult(HttpContext);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static ActionResult UnauthorizedUrlRequestResult(HttpContextBase context)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string url = context.Request.Url.ToString();
                int index = url.IndexOf('?');
                if (index >= 0)
                    url = url.Substring(0, index);
                string message = string.Format(Strings.ForbiddenPageContent, url);
                using (var controller = new UnauthorizedController())
                {
                    return controller.PermissionAuth(message);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static ActionResult UnauthorizedWriteRequestResult(string facilityName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string message = string.Format(Strings.ForbiddenWriteAttempt, facilityName);
                using (var controller = new UnauthorizedController())
                {
                    return controller.PermissionAuth(message);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the date time now.
        /// </summary>
        public static DateTime DateTimeNowOnCurrentFacility
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    return ConvertDateTimeForCurrentFacility(DateTime.UtcNow);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Converts the date time for current facility.
        /// </summary>
        /// <param name="datetime">The datetime.</param>
        /// <returns></returns>
        public static DateTime ConvertDateTimeForCurrentFacility(DateTime datetime)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return TimeZoneInfo.ConvertTime(datetime, TimeZoneInfoOnCurrentFacility);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the time zone info on current facility.
        /// </summary>
        public static TimeZoneInfo TimeZoneInfoOnCurrentFacility
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    return FacadeUtil.GetFacilityTimeZoneInfo(FacadeManager.UserInterface.GetProfile().Facility.Id);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        /// <summary>
        /// Gets the facility time zone code from configurations.
        /// </summary>
        /// <returns>Facility Time Zone Code or empty string.</returns>
        public string GetFacilityTimeZone()
        {
            FacilitySettings facilitySettings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);
            if (facilitySettings != null && facilitySettings.FacilitySiteTimeZone != null)
                return facilitySettings.FacilitySiteTimeZone.code;
            else
                return string.Empty;
        }

        public static string RepTimeZoneMinutesOffset
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    DateTime dtUtc = DateTime.UtcNow;
                    DateTime dtTz = TimeZoneInfo.ConvertTimeFromUtc(dtUtc, TimeZoneInfoOnCurrentFacility);
                    return dtTz.Subtract(dtUtc).TotalMinutes.ToString();
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }

        protected override void OnActionExecuting(ActionExecutingContext filterContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    User user = FacadeManager.UserInterface.GetProfile();
                }
                catch (UserProfileEmptyException)
                {
                    filterContext.Result = new EmptyResult();
                    RedirectLogin(filterContext.HttpContext);
                    return;
                }
                base.OnActionExecuting(filterContext);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

#if !DEBUG
        /// <summary>
        /// Called after the action method is invoked.
        /// On SecurityNegotiationException force a re-logon.
        /// </summary>
        /// <param name="filterContext">Information about the current request and action.</param>
        protected sealed override void OnActionExecuted(ActionExecutedContext filterContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (IsSessionCriticalException(filterContext.Exception))
                {
                    RedirectLogin(filterContext.HttpContext);
                    return;
                }

                ActionResult ar = OnException(filterContext.Exception);
                if (ar != null)
                {
                    filterContext.Result = ar;
                    filterContext.ExceptionHandled = true;
                    return;
                }
                OnCustomActionExecuted(filterContext);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
#endif

        /// <summary>
        /// Checks whether the raised exception causes the current session to be abandoned.
        /// </summary>
        /// <param name="exception">The exception to check.</param>
        /// <returns>Boolean value of true if the session should be abandoned, false otherwise.</returns>
        bool IsSessionCriticalException(Exception exception)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (exception is SecurityNegotiationException)
                    return true;
                if (exception is UserProfileEmptyException)
                    return true;
                return false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Redirects to login page using the given HttpContext.
        /// </summary>
        /// <param name="httpContext">HttpContextBase used to redirect.</param>
        void RedirectLogin(HttpContextBase httpContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                loggedUser = null;
                System.Web.Security.FormsAuthentication.SignOut();
                httpContext.Session.Abandon();
                string url = httpContext.Request.Url.AbsoluteUri;
                if (url.IndexOf(@"/EMSMobile") >= 0)
                    httpContext.Response.Redirect("~/EMSMobileLogon?code=" + httpContext.Request.Params["FacilityCode"]);
                else if (url.IndexOf(@"/WardWhiteBoardAjax") >= 0 || url.IndexOf(@"/WardWhiteboard/WardWhiteBoard") >= 0)
                    RedirectWhiteboardKiosk(httpContext);
                else
                    httpContext.Response.Redirect("~/Account/LogOn");
            }
            catch (HttpException) { }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void RedirectWhiteboardKiosk(HttpContextBase httpContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder url = new StringBuilder();
                if (httpContext.Request.Params["model"] != null)
                {
                    JavaScriptSerializer jss = new JavaScriptSerializer();
                    WardWhiteboardViewModel input = jss.Deserialize<WardWhiteboardViewModel>(httpContext.Request.Params["model"]);
                    if (input.IsFromWhiteboardKiosk == "1")
                    {
                        url.Append("~/WardWhiteboardUrl?facilityCode=");
                        url.Append(input.FacilityCode);
                        url.Append("&wardName=");
                        if (string.IsNullOrEmpty(input.DivisionExtension) || input.DivisionExtension.Equals(Guid.Empty.ToString()))
                            url.Append(Constants.ALL);
                        else
                        {
                            string domain = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                            Division division = BMS.ServicesWrapper.EIS.EISFactory.InstanceWindows.GetDivision(new II(domain, input.DivisionExtension));
                            if (division != null && !string.IsNullOrEmpty(division.Name))
                                url.Append(division.Name);
                            else
                                url.Append(Constants.ALL);
                        }
                        url.Append("&splitScreen=");
                        url.Append(input.SplitScreen);
                        url.Append("&displayPTCode=");
                        url.Append(input.DisplayPTCode);
                        url.Append("&genderColorCode=");
                        url.Append(input.GenderColorCode);
                        url.Append("&sortBy=");
                        url.Append(input.SortBySelected);
                        url.Append("&displayFooterCensus=");
                        url.Append(input.DisplayFooterCensus);
                        url.Append("&displayStaffAttending=");
                        url.Append(input.DisplayStaffAttendingSelected);
                        url.Append("&scrollRate=");
                        url.Append(input.ScrollRateSelected);
                        httpContext.Response.Redirect(url.ToString());
                    }
                    else
                        httpContext.Response.Redirect("~/Account/LogOn");
                }
                else
                {
                    if (httpContext.Request.Params["IsFromWhiteboardKiosk"] == "1")
                    {
                        url.Append("~/WardWhiteboardUrl?facilityCode=");
                        url.Append(httpContext.Request.Params["FacilityCode"]);
                        url.Append("&wardName=");
                        if (string.IsNullOrEmpty(httpContext.Request.Params["DivisionExtension"]) || httpContext.Request.Params["DivisionExtension"].Equals(Guid.Empty.ToString()))
                            url.Append(Constants.ALL);
                        else
                        {
                            string domain = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                            Division division = BMS.ServicesWrapper.EIS.EISFactory.InstanceWindows.GetDivision(new II(domain, httpContext.Request.Params["DivisionExtension"]));
                            if (division != null && !string.IsNullOrEmpty(division.Name))
                                url.Append(division.Name);
                            else
                                url.Append(Constants.ALL);
                        }
                        url.Append("&splitScreen=");
                        url.Append(httpContext.Request.Params["SplitScreen"]);
                        url.Append("&displayPTCode=");
                        url.Append(httpContext.Request.Params["DisplayPTCode"]);
                        url.Append("&genderColorCode=");
                        url.Append(httpContext.Request.Params["GenderColorCode"]);
                        url.Append("&sortBy=");
                        url.Append(httpContext.Request.Params["SortBySelected"]);
                        url.Append("&displayFooterCensus=");
                        url.Append(httpContext.Request.Params["DisplayFooterCensus"]);
                        url.Append("&displayStaffAttending=");
                        url.Append(httpContext.Request.Params["DisplayStaffAttendingSelected"]);
                        url.Append("&scrollRate=");
                        url.Append(httpContext.Request.Params["ScrollRateSelected"]);
                        httpContext.Response.Redirect(url.ToString());
                    }
                    else
                        httpContext.Response.Redirect("~/Account/LogOn");
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Called when [custom action executed]. Virtual method for override from Controller.
        /// </summary>
        /// <param name="filterContext">The filter context.</param>
        protected virtual void OnCustomActionExecuted(ActionExecutedContext filterContext)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                base.OnActionExecuted(filterContext);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        /// <summary>
        /// Called when [exception]. Virtual method for override from Controller.
        /// </summary>
        /// <param name="exception">The exception.</param>
        /// <returns></returns>
        protected virtual ActionResult OnException(Exception exception)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                ActionResult result = exception == null ? null :
                              OnBaseFacadeException(exception as BaseFacadeException) ??
                              OnFaultCustomSqlExecption(exception as FaultException<CustomSQLException>) ??
                              OnFaultGenericWfServiceFault(exception as FaultException<GenericWFServiceFault>) ??
                              OnBaseFacadeException(new BaseFacadeException(exception));

                return result;

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private bool GetDisplayDetailedErrorMessage()
        {
            return Convert.ToBoolean(System.Configuration.ConfigurationManager.AppSettings["DisplayDetailedErrorMessage"]);
        }

        private ActionResult OnBaseFacadeException(BaseFacadeException exception)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (exception == null)
                    return null;
                Tracer.TraceException(exception);
                FaultException<GenericWFServiceFault> ex = exception.InnerException as FaultException<GenericWFServiceFault>;
                if (!GetDisplayDetailedErrorMessage() && ex == null)
                    exception = new BaseFacadeException(new Exception(Strings.HideErrorMessage));
                TempData.Put<BaseFacadeException>(exception);
                return new RedirectResult("~/Exception/Index");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private ActionResult OnFaultGenericWfServiceFault(FaultException<GenericWFServiceFault> fault)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (fault == null || fault.Detail == null)
                    return null;

                WFInvalidOperation wfEx = fault.Detail.ServiceFault as WFInvalidOperation;
                if (wfEx != null && (wfEx.FriendlyMessage == Resources.WF_INVALID_OPERATION || wfEx.FriendlyMessage == Resources.WF_INVALID_OPERATION_BED))
                {
                    StackTrace stackTrace = new StackTrace((System.Exception)(fault), false);

                    foreach (StackFrame frame in stackTrace.GetFrames())
                    {
                        MethodBase method = frame.GetMethod();
                        if (method.DeclaringType == null)
                            continue;
                        if (method.DeclaringType.FullName == typeof(BMS.ServicesWrapper.Proxy.WF.BMSWorkflowsWFIWaitingListFlowClient).FullName)
                            switch (method.Name)
                            {
                                case BMS.Utils.Constants.WAITING_LIST_FLOW_CREATE_ITEM:
                                    wfEx.FriendlyMessage = Resources.WF_WAITING_LIST_ERROR_ALREADY_EXISTS_ON_CREATE;
                                    break;
                                case BMS.Utils.Constants.WAITING_LIST_FLOW_UNDO_REMOVE_ITEM:
                                    wfEx.FriendlyMessage = Resources.WF_WAITING_LIST_ERROR_ALREADY_EXISTS_ON_UNDO_REMOVE;
                                    break;
                            }

                        if (method.DeclaringType.FullName == typeof(BMS.ServicesWrapper.Proxy.WF.BMSWorkflowsWFITransferFlowClient).FullName &&
                            method.Name == BMS.Utils.Constants.TRANSFER_FLOW_CREATE_REQUEST)
                            wfEx.FriendlyMessage = Resources.WF_TRANSFER_ERROR_ALREADY_EXISTS;
                        if (method.DeclaringType.FullName == typeof(BMS.ServicesWrapper.Proxy.WF.BMSWorkflowsWFIBedUnavailableFlowClient).FullName &&
                            method.Name == BMS.Utils.Constants.BED_UNAVAILABLE_CREATE)
                            wfEx.FriendlyMessage = Resources.WF_BED_UNAVAILABLE_ERROR_ALREADY_EXISTS;
                    }
                    return OnBaseFacadeException(fault.Detail.ServiceFault as BaseFacadeException);
                }

                WfOtherException wfOtherException = fault.Detail.ServiceFault as WfOtherException;
                if (wfOtherException != null && wfOtherException.ErrorMessage.Contains(Resources.WF_ERROR_INSTANCE_NOT_FOUND_PART))
                {
                    StackTrace stackTrace = new StackTrace((System.Exception)(fault), false);

                    foreach (StackFrame frame in stackTrace.GetFrames())
                    {
                        MethodBase method = frame.GetMethod();
                        if (method.DeclaringType == null)
                            continue;
                    }
                }
                return OnBaseFacadeException(fault.Detail.ServiceFault as BaseFacadeException);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private ActionResult OnCustomSqlExeption(CustomSQLException exception)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (exception == null)
                    return null;

                if (exception.SPName == "_CHECK_UPDATE_PERMISSION")
                {
                    if (exception.ErrorMessage == "Acces denied: \"BMS, Read\"" || exception.ErrorMessage == "Acces denied: \"BMS, Write\"")
                    {
                        string message = exception.ErrorMessage + " on current facility";
                        using (var controller = new UnauthorizedController())
                        {
                            return controller.PermissionAuth(message);
                        }

                    }
                }

                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private ActionResult OnFaultCustomSqlExecption(FaultException<CustomSQLException> fault)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (fault == null)
                    return null;
                return OnCustomSqlExeption(fault.Detail);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Filter attribute used to set the loggedUser before the action method executes on the 
        /// specified controller.
        /// </summary>
        public class LoggedUserFilterAttribute : ActionFilterAttribute
        {
            /// <summary>
            /// Called by the ASP.NET MVC framework before the action method executes.
            /// Sets the loggedUser before the action method executes.
            /// </summary>
            /// <param name="filterContext">The filter context.</param>
            public override void OnActionExecuting(ActionExecutingContext filterContext)
            {
                BaseController controller = filterContext.Controller as BaseController;
                if (controller != null)
                {
                    controller.loggedUser = FacadeManager.UserInterface.GetProfile();
                    if (controller.loggedUser != null)
                    {
                        AntiXssEncoder xss = new AntiXssEncoder();
                        controller.ViewBag.FacilityName = controller.loggedUser.Facility.Name = xss.Decode(controller.loggedUser.Facility.Name);
                        controller.ViewBag.FacilityCode = controller.loggedUser.Facility.Code;
                        controller.QueryStrings = GetControllerActionParameters(controller);
                    }
                }
                base.OnActionExecuting(filterContext);
            }

            private Dictionary<string, string> GetControllerActionParameters(BaseController controller)
            {
                Dictionary<string, string> result = new Dictionary<string, string>();
                string url = controller.Request.Url.ToString();
                if (url.Contains('?'))
                {
                    url = url.Substring(url.IndexOf('?') + 1);
                    url = CustomEncryption.Decrypt(url.Substring(2), controller.loggedUser.Salt);
                    string[] parameters = url.Split('&');
                    foreach (string p in parameters)
                    {
                        //HHG INC000001259968: Replace @amp with "&" in parmeter of query string;   
                        string tempp = p.Replace("@amp;", "&");
                        result.Add(tempp.Substring(0, tempp.IndexOf('=')), tempp.Substring(tempp.IndexOf('=') + 1));
                    }

                }
                return result;
            }
        }

        public static string EncryptQueryString(string[] names, string[] values, string salt)
        {
            StringBuilder sb = new StringBuilder();
            if (names.Length != values.Length)
                return null;

            for (int i = 0; i < names.Length; i++)
            {
                sb.Append(names[i]);
                sb.Append("=");

                //HHG INC000001259968: Replace "&" with @amp; in query string           
                if (values[i] != null)
                {
                    if (values[i].IndexOf("&") > 0)
                    {
                        sb.Append(values[i].Replace("&", "@amp;"));
                    }
                    else
                    {
                        sb.Append(values[i]);
                    }
                }

                sb.Append("&");
            }

            if (sb.Length > 0)
            {
                sb.Length--;
                return CustomEncryption.Encrypt(sb.ToString(), salt);
            }
            return null;
        }

        [AttributeUsage(AttributeTargets.Class | AttributeTargets.Method, AllowMultiple = false, Inherited = true)]
        public class ValidateAntiForgeryTokenWrapperAttribute : FilterAttribute, IAuthorizationFilter
        {
            private ValidateAntiForgeryTokenAttribute _validator;

            private readonly AcceptVerbsAttribute _verbs;

            public ValidateAntiForgeryTokenWrapperAttribute(HttpVerbs verbs)
            {
                this._verbs = new AcceptVerbsAttribute(verbs);
            }

            public void OnAuthorization(AuthorizationContext filterContext)
            {
                string httpMethodOverride = filterContext.HttpContext.Request.GetHttpMethodOverride();
                string url = filterContext.HttpContext.Request.Url.AbsoluteUri;
                if (this._verbs.Verbs.Contains(httpMethodOverride, StringComparer.OrdinalIgnoreCase))
                {
                    BaseController controller = filterContext.Controller as BaseController;
                    if (controller != null)
                    {
                        if (controller.loggedUser == null)
                            controller.loggedUser = FacadeManager.UserInterface.GetProfile();
                        if (string.IsNullOrEmpty(controller.loggedUser.Salt))
                            controller.loggedUser.Salt = Utils.CustomEncryption.GetRandomSalt(controller.loggedUser.UserName);

                        this._validator = new ValidateAntiForgeryTokenAttribute();
                    }
                    try
                    {
                        var request = filterContext.HttpContext.Request;

                        //  Only validate POSTs
                        //if (request.HttpMethod == WebRequestMethods.Http.Post)
                        {
                            //  Ajax POSTs and normal form posts have to be treated differently when it comes
                            //  to validating the AntiForgeryToken
                            if (request.IsAjaxRequest())
                            {
                                var antiForgeryCookie = request.Cookies[AntiForgeryConfig.CookieName];

                                var cookieValue = antiForgeryCookie != null
                                    ? antiForgeryCookie.Value
                                    : null;

                                AntiForgery.Validate(cookieValue, request.Headers["__RequestVerificationToken"]);
                            }
                            else
                            {
                                new ValidateAntiForgeryTokenAttribute().OnAuthorization(filterContext);
                            }
                        }
                        //Validate(this._validator, filterContext.HttpContext, controller.loggedUser.Salt);
                        //this._validator.OnAuthorization(filterContext);
                    }
                    catch (Exception ex)
                    {
                        bool throwEx = true;
                        if (ex.Message == "A required anti-forgery token was not supplied or was invalid.")
                        {
                            if (!string.IsNullOrEmpty(url) && (url.IndexOf(@"/WardWhiteBoardAjax") >= 0 || url.IndexOf(@"/WardWhiteboard/WardWhiteBoard") >= 0))
                            {
                                if (filterContext.HttpContext.Request.Params["model"] != null)
                                {
                                    JavaScriptSerializer jss = new JavaScriptSerializer();
                                    WardWhiteboardViewModel input = jss.Deserialize<WardWhiteboardViewModel>(filterContext.HttpContext.Request.Params["model"]);
                                    if (input.IsFromWhiteboardKiosk == "1")
                                        throwEx = false;
                                }
                                else if (filterContext.HttpContext.Request.Params["IsFromWhiteboardKiosk"] == "1")
                                    throwEx = false;
                            }
                            if (controller != null && throwEx)
                                Tracer.TraceMessage("CSRF Exception on " + controller.Request.CurrentExecutionFilePath + " for user " + (controller.loggedUser != null ? controller.loggedUser.UserName : "NULL"));
                        }
                        if (throwEx)
                            throw;
                    }
                }
            }

            private void Validate(ValidateAntiForgeryTokenAttribute _validator, HttpContextBase context, string salt)
            {
                string antiForgeryTokenName = AntiForgeryData.GetAntiForgeryTokenName(null);
                string str2 = AntiForgeryData.GetAntiForgeryTokenName(context.Request.ApplicationPath);
                HttpCookie cookie = context.Request.Cookies[str2];
                if ((cookie == null) || string.IsNullOrEmpty(cookie.Value))
                {
                    Tracer.TraceMessage("ValidateAntiForgeryToken - cookie null or empty");
                    throw CreateValidationException();
                }
                AntiForgeryDataSerializer serializer = new AntiForgeryDataSerializer();
                AntiForgeryData data = serializer.Deserialize(cookie.Value);
                string str3 = context.Request.Form[antiForgeryTokenName];
                if (string.IsNullOrEmpty(str3))
                {
                    Tracer.TraceMessage("ValidateAntiForgeryToken - antiforgery token from form is null or empty");
                    throw CreateValidationException();
                }
                AntiForgeryData data2 = serializer.Deserialize(str3);
                if (!string.Equals(data.Value, data2.Value, StringComparison.Ordinal))
                {
                    Tracer.TraceMessage("ValidateAntiForgeryToken - value from cookie is different than the value from form. Form value: " + data2.Value + " Cookie value: " + data.Value);
                    throw CreateValidationException();
                }
                string username = AntiForgeryData.GetUsername(context.User);
                if (!string.Equals(data2.Username, username, StringComparison.OrdinalIgnoreCase))
                {
                    Tracer.TraceMessage("ValidateAntiForgeryToken - user name does not match. Form user: " + data2.Username + " Context user: " + username);
                    throw CreateValidationException();
                }
                if (!string.Equals(salt ?? string.Empty, data2.Salt, StringComparison.Ordinal))
                {
                    Tracer.TraceMessage("ValidateAntiForgeryToken - salt does not match. Form salt: " + data2.Salt + " Logged user salt: " + salt);
                    throw CreateValidationException();
                }
            }

            private static HttpAntiForgeryException CreateValidationException()
            {
                return new HttpAntiForgeryException("A required anti-forgery token was not supplied or was invalid.");
            }
        }

        internal sealed class AntiForgeryData
        {
            // Fields
            private DateTime _creationDate;
            private static readonly System.Security.Cryptography.RNGCryptoServiceProvider _prng = new System.Security.Cryptography.RNGCryptoServiceProvider();
            private string _salt;
            private string _username;
            private string _value;
            private const string AntiForgeryTokenFieldName = "__RequestVerificationToken";
            private const int TokenLength = 0x10;

            // Methods
            public AntiForgeryData()
            {
                this._creationDate = DateTime.UtcNow;
            }

            public AntiForgeryData(AntiForgeryData token)
            {
                this._creationDate = DateTime.UtcNow;
                if (token == null)
                {
                    throw new ArgumentNullException("token");
                }
                this.CreationDate = token.CreationDate;
                this.Salt = token.Salt;
                this.Username = token.Username;
                this.Value = token.Value;
            }

            private static string Base64EncodeForCookieName(string s)
            {
                return Convert.ToBase64String(Encoding.UTF8.GetBytes(s)).Replace('+', '.').Replace('/', '-').Replace('=', '_');
            }

            private static string GenerateRandomTokenString()
            {
                byte[] data = new byte[0x10];
                _prng.GetBytes(data);
                return Convert.ToBase64String(data);
            }

            internal static string GetAntiForgeryTokenName(string appPath)
            {
                if (string.IsNullOrEmpty(appPath))
                {
                    return "__RequestVerificationToken";
                }
                return ("__RequestVerificationToken_" + Base64EncodeForCookieName(appPath));
            }

            internal static string GetUsername(System.Security.Principal.IPrincipal user)
            {
                if (user != null)
                {
                    System.Security.Principal.IIdentity identity = user.Identity;
                    if ((identity != null) && identity.IsAuthenticated)
                        return identity.Name;
                }
                return string.Empty;
            }

            public static AntiForgeryData NewToken()
            {
                string str = GenerateRandomTokenString();
                return new AntiForgeryData { Value = str };
            }

            // Properties
            public DateTime CreationDate
            {
                get { return this._creationDate; }
                set { this._creationDate = value; }
            }

            public string Salt
            {
                get { return (this._salt ?? string.Empty); }
                set { this._salt = value; }
            }

            public string Username
            {
                get { return (this._username ?? string.Empty); }
                set { this._username = value; }
            }

            public string Value
            {
                get { return (this._value ?? string.Empty); }
                set { this._value = value; }
            }
        }

        internal class AntiForgeryDataSerializer
        {
            // Fields
            internal Func<string, byte[]> Decoder = value => Unprotect(Base64ToHex(value));
            internal Func<byte[], string> Encoder = bytes => HexToBase64(Protect(bytes).ToUpperInvariant());

            const string MachineKeyPurpose = "MyApp:Username:{0}";
            const string Anonymous = "<anonymous>";

            private static string Base64ToHex(string base64)
            {
                StringBuilder builder = new StringBuilder(base64.Length * 4);
                foreach (byte num in Convert.FromBase64String(base64))
                {
                    builder.Append(HexDigit(num >> 4));
                    builder.Append(HexDigit(num & 15));
                }
                return builder.ToString();
            }

            private static string GetMachineKeyPurpose(IPrincipal user)
            {
                return String.Format(MachineKeyPurpose, user.Identity.IsAuthenticated ? user.Identity.Name : Anonymous);
            }

            private static string Protect(byte[] data)
            {
                if (data == null || data.Length == 0) return null;
                var purpose = GetMachineKeyPurpose(Thread.CurrentPrincipal);
                var value = System.Web.Security.MachineKey.Protect(data, purpose);
                return Convert.ToBase64String(value);
            }

            private static byte[] Unprotect(string value)
            {
                if (String.IsNullOrWhiteSpace(value)) return null;
                var purpose = GetMachineKeyPurpose(Thread.CurrentPrincipal);
                var bytes = Convert.FromBase64String(value);
                return System.Web.Security.MachineKey.Unprotect(bytes, purpose);
            }

            public virtual AntiForgeryData Deserialize(string serializedToken)
            {
                AntiForgeryData data2;
                if (string.IsNullOrEmpty(serializedToken))
                {
                    throw new ArgumentException("Argument cannot be null or empty", "serializedToken");
                }
                try
                {
                    using (System.IO.MemoryStream stream = new System.IO.MemoryStream(this.Decoder(serializedToken)))
                    {
                        using (System.IO.BinaryReader reader = new System.IO.BinaryReader(stream))
                        {
                            AntiForgeryData data = new AntiForgeryData
                            {
                                Salt = reader.ReadString(),
                                Value = reader.ReadString(),
                                CreationDate = new DateTime(reader.ReadInt64()),
                                Username = reader.ReadString()
                            };
                            data2 = data;
                        }
                    }
                }
                catch (Exception exception)
                {
                    Tracer.TraceMessage("AntiForgeryDataSerializer Deserialize exception");
                    Tracer.TraceException(exception);
                    throw new HttpAntiForgeryException("A required anti-forgery token was not supplied or was invalid.", exception);
                }
                return data2;
            }

            private static char HexDigit(int value)
            {
                return ((value > 9) ? ((char)(value + 0x37)) : ((char)(value + 0x30)));
            }

            private static string HexToBase64(string hex)
            {
                int num = hex.Length / 2;
                byte[] inArray = new byte[num];
                for (int i = 0; i < num; i++)
                {
                    inArray[i] = (byte)((HexValue(hex[i * 2]) << 4) + HexValue(hex[(i * 2) + 1]));
                }
                return Convert.ToBase64String(inArray);
            }

            private static int HexValue(char digit)
            {
                if (digit <= '9')
                {
                    return (digit - '0');
                }
                return (digit - '7');
            }

            public virtual string Serialize(AntiForgeryData token)
            {
                string str;
                if (token == null)
                {
                    throw new ArgumentNullException("token");
                }
                using (System.IO.MemoryStream stream = new System.IO.MemoryStream())
                {
                    using (System.IO.BinaryWriter writer = new System.IO.BinaryWriter(stream))
                    {
                        writer.Write(token.Salt);
                        writer.Write(token.Value);
                        writer.Write(token.CreationDate.Ticks);
                        writer.Write(token.Username);
                        str = this.Encoder(stream.ToArray());
                    }
                }
                return str;
            }
        }

        /// <summary>
        /// Using to orderby numbers that are string.
        /// </summary>
        public class NumericComparer : IComparer<string>
        {
            public int Compare(string s1, string s2)
            {
                if (IsNumeric(s1) && IsNumeric(s2))
                {
                    if (Convert.ToInt32(s1) > Convert.ToInt32(s2))
                        return 1;
                    if (Convert.ToInt32(s1) < Convert.ToInt32(s2))
                        return -1;
                    if (Convert.ToInt32(s1) == Convert.ToInt32(s2))
                        return 0;
                }

                if (IsNumeric(s1) && !IsNumeric(s2))
                    return -1;

                if (!IsNumeric(s1) && IsNumeric(s2))
                    return 1;

                return string.Compare(s1, s2, true);
            }

            public static bool IsNumeric(object value)
            {
                try
                {
                    int i = Convert.ToInt32(value.ToString());
                    return true;
                }
                catch (FormatException)
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// Loads the standardized comments.
        /// </summary>
        /// <param name="displayName">The display name.</param>
        /// <returns></returns>
        [HttpPost]
        public JsonResult LoadStandardComments(string displayName)
        {
            displayName = Encoding.UTF8.GetString(Convert.FromBase64String(displayName));
            Array StandardList;
            List<CD> comments = new List<CD>();
            comments = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Comments);
            if (comments != null)
                if (displayName != null)
                    StandardList = comments.Where(a => a.displayName.StartsWith(displayName.ToUpper())).OrderBy(a => a.displayName).Select(a => a.displayName).ToArray();
                else
                    StandardList = null;
            else
                StandardList = null;

            return Json(StandardList);
        }

        #region Translator SisterGroup

        public void GetSistersIdentifierFromCodes([Bind(Exclude = "RememberMe")] out SisterSitesGroup outGroup, out string codesDoNotExist, [Bind(Exclude = "RememberMe")] SisterSitesGroup group)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string identifiers = string.Empty;
                codesDoNotExist = string.Empty;
                outGroup = new SisterSitesGroup();
                if (group != null)
                {
                    if (!string.IsNullOrWhiteSpace(group.SiteCodeList))
                    {
                        string[] codes = group.SiteCodeList.Split(',');
                        foreach (string codeFacility in codes)
                        {
                            if (!string.IsNullOrWhiteSpace(codeFacility))
                            {
                                Facility facility = FacadeManager.EntityInterface.GetFacilityByCode(codeFacility.Trim());
                                if (facility != null)
                                {
                                    if (string.IsNullOrEmpty(identifiers))
                                        identifiers = facility.Id.extension;
                                    else
                                        identifiers += ", " + facility.Id.extension;
                                }
                                else
                                {
                                    if (string.IsNullOrWhiteSpace(codesDoNotExist))
                                        codesDoNotExist = codeFacility;
                                    else
                                        codesDoNotExist += ", " + codeFacility;
                                }
                            }
                        }
                    }
                    outGroup.SiteCodeList = identifiers;
                    outGroup.RecordNumber = group.RecordNumber;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public SisterSitesGroup GetSistersCodeFromIdentifier([Bind(Exclude = "RememberMe")] SisterSitesGroup group)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (group != null)
                {
                    string codes = string.Empty;
                    if (!string.IsNullOrWhiteSpace(group.SiteCodeList))
                    {
                        string[] identifiers = group.SiteCodeList.Split(',');
                        foreach (string identifierFacility in identifiers)
                        {
                            Facility facility = FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, identifierFacility.Trim()));
                            if (facility != null)
                                if (string.IsNullOrEmpty(codes))
                                    codes = facility.Code;
                                else
                                    codes += ", " + facility.Code;

                        }

                    }
                    group.SiteCodeList = codes;
                }
                return group;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Evacuation Patient

        public void AddPatientIntoWaitingList([Bind(Exclude = "RememberMe")] Patient patient, [Bind(Exclude = "RememberMe")] Guid flowId, [Bind(Exclude = "RememberMe")] CD waitingArea, [Bind(Exclude = "RememberMe")] CD evacDispositionStatus, [Bind(Exclude = "RememberMe")] CD evacTransportationProvider)
        {

            WaitingListItem waitingListItem = new WaitingListItem()
            {
                Id = new II(this.loggedUser.Domain, null),
                CreationDate = DateTimeNowOnCurrentFacility,
                Patient = patient,
                LastEditBy = this.loggedUser.UserName,
                IsEvacuationPatient = true,
                FlowId = flowId,
                Facility = this.loggedUser.Facility,
                VistaSite = this.loggedUser.Facility.VistaSite,
                WaitingArea = waitingArea,
                Problem = patient.AdmittingDiagnosis,
                EvacuationDispositionStatus = evacDispositionStatus,
                EvacuationTransportationProvider = evacTransportationProvider
            };

            try
            {
                FacadeManager.WorkflowInterface.CreateWaitingListItem(waitingListItem, loggedUser.Facility);
            }
            catch (FaultException<GenericWFServiceFault> fault)
            {
                WFInvalidOperation wfEx = fault.Detail.ServiceFault as WFInvalidOperation;
                if (wfEx != null && wfEx.FriendlyMessage == Resources.WF_INVALID_OPERATION)
                {
                    StackTrace stackTrace = new StackTrace((System.Exception)(fault), false);

                    foreach (StackFrame frame in stackTrace.GetFrames())
                    {
                        MethodBase method = frame.GetMethod();
                        if (method.DeclaringType == null)
                            continue;
                        if (method.DeclaringType.FullName == typeof(BMS.ServicesWrapper.Proxy.WF.BMSWorkflowsWFIWaitingListFlowClient).FullName && method.Name == BMS.Utils.Constants.WAITING_LIST_FLOW_CREATE_ITEM)
                        {
                            WaitingListItem wli = FacadeManager.ADTInterface.GetWaitingListItem(null, patient.Id.extension);
                            if (wli.IsEvacuationPatient != true)
                            {
                                wli.IsEvacuationPatient = true;
                                FacadeManager.WorkflowInterface.ModifyWaitingListItem(wli, loggedUser.Facility);
                            }
                            break;
                        }
                    }
                }
            }
        }
        #endregion

        #region User Configuration Section

        public User GetUser(string id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                User user = string.IsNullOrEmpty(id) || id == "Default"
                                                     ? FacadeManager.UserInterface.GetDefaultUserProfile() :
                                                       FacadeManager.UserInterface.GetUser(id);
                II userId = user.Id;
                user = FacadeManager.ConfigurationInterface.GetUserProfile(user.UserName, user.Domain);
                user.Id = userId;
                return user;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        #endregion

        #region Background Processors

        public List<SelectableEnum<DataRetrievalMethod>> GetRetrievalMethods()
        {
            return new List<DataRetrievalMethod>((DataRetrievalMethod[])Enum.GetValues(typeof(DataRetrievalMethod)).Cast<DataRetrievalMethod>().ToArray())
                                .Select(drm =>
                                {
                                    if (DataRetrievalMethod.VIA.Equals(drm))
                                        return new SelectableEnum<DataRetrievalMethod>(drm, true);
                                    else
                                        return new SelectableEnum<DataRetrievalMethod>(drm, false);
                                }
                                        )
                                .ToList();
        }


        public List<SelectableEnum<VI.Admin.VistaDataType>> GetDataTypesList()
        {
            List<VI.Admin.VistaDataType> dataTypesList = new List<VI.Admin.VistaDataType>((VI.Admin.VistaDataType[])Enum.GetValues(typeof(VI.Admin.VistaDataType)));
            dataTypesList.Remove(VI.Admin.VistaDataType.None);
            return dataTypesList.Select(vdt => new SelectableEnum<VI.Admin.VistaDataType>(vdt, false)).ToList();
        }

        public List<SelectableEnum<JobStatus>> GetJobStatus()
        {
            return new List<JobStatus>((JobStatus[])Enum.GetValues(typeof(JobStatus))).Select(js => new SelectableEnum<JobStatus>(js, true)).ToList();
        }

        public JobLogModel GetLogModel([Bind(Exclude = "RememberMe")] JobLogInfo info, Dictionary<string, string> timeZones)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string tz = string.Empty;
                if (timeZones.ContainsKey(info.VistaId.ToLower()))
                    tz = timeZones[info.VistaId.ToLower()];
                return new JobLogModel()
                {
                    EndDate = info.EndDate.HasValue ? info.EndDate.Value.ToString(Strings.HHssDateTime) + " " + tz : string.Empty,
                    ErrorMessage = info.ErrorMessage ?? string.Empty,
                    Details = info.Details ?? string.Empty,
                    LaunchType = SplitCamelCase(info.LaunchType.ToString("G")),
                    Parameters = GetParameters(info.Parameters),
                    RetrievalMethod = SplitCamelCase(info.RetrievalMethod.ToString("G")),
                    RetrievedDataType = SplitCamelCase(info.RetrievedDataType.ToString("G")),
                    RetrievedEntriesNumber = info.RetrievedEntriesNumber != 0 ? info.RetrievedEntriesNumber + "" : string.Empty,
                    StartDate = info.StartDate.ToString(Strings.HHssDateTime) + " " + tz,
                    Status = SplitCamelCase(info.Status.ToString("G")),
                    VistaId = info.VistaId,
                    VistaName = info.VistaName,
                    TotalCount = info.TotalCount,
                    Index = info.Index,
                    EndDateTime = info.EndDate
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string SplitCamelCase(string s)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return Regex.Replace(Regex.Replace(s, @"(\P{Ll})(\P{Ll}\p{Ll})", "$1 $2"), @"(\p{Ll})(\P{Ll})", "$1 $2");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public string GetParameters(Dictionary<string, object> dictionary)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string result = string.Empty;
                if (dictionary != null)
                {
                    foreach (KeyValuePair<string, object> kvp in dictionary)
                        result += "(" + kvp.Key + ", " + (kvp.Value != null ? kvp.Value.ToString() : "null") + ")";
                }

                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public VI.VistASite GetVistaSite(string vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                foreach (VI.VistASite vs in FacadeManager.ConfigurationInterface.GetVistaSites(loggedUser.Domain))
                    if (vs.Id == vistaSiteId)
                        return vs;
                return new VI.VistASite() { Name = string.Empty };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public VI.Admin.JobCalendar GetScheduler(string schedulerId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                foreach (VI.Admin.JobCalendar calendar in FacadeManager.ConfigurationInterface.GetDefinedCalendars(loggedUser.Domain))
                    if (calendar.Id == schedulerId)
                        return calendar;
                return new VI.Admin.JobCalendar()
                {
                    Name = string.Empty,
                    EndTime = new TimeSpan(23, 59, 0),
                    Recurs = 1
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        private string GetFullUserName()
        {
            if (loggedUser == null)
                return null;

            string[] split = loggedUser.UserName.Split(new char[] { '\\' });
            string fullUserName = split[1] + "@" + split[0];
            return fullUserName;
        }
    }
}
